<?php
/**
 * Contact List
 *
 * @package     AutomatorWP\Integrations\FluentCRM\Actions\Contact_List
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_FluentCRM_Contact_List extends AutomatorWP_Integration_Action {

    public $integration = 'fluentcrm';
    public $action = 'fluentcrm_contact_list';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add or remove contact to list', 'automatorwp-pro' ),
            'select_option'     => __( 'Add or remove contact to <strong>list</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Contact Email. %3$s: List. */
            'edit_label'        => sprintf( __( '%1$s %2$s to %3$s', 'automatorwp-pro' ), '{operation}', '{contact}', '{list}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Contact Email. %3$s: List. */
            'log_label'         => sprintf( __( '%1$s %2$s to %3$s', 'automatorwp-pro' ), '{operation}', '{contact}', '{list}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'add'       => __( 'Add', 'automatorwp-pro' ),
                                'remove'    => __( 'Remove', 'automatorwp-pro' ),
                            ),
                            'default' => 'add'
                        ),
                    )
                ),
                'contact' => array(
                    'from' => 'contact',
                    'default' => __( 'contact', 'automatorwp-pro' ),
                    'fields' => array(
                        'contact' => array(
                            'name' => __( 'Contact Email:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                    )
                ),
                'list' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'list',
                    'option_default'    => __( 'Select a list', 'automatorwp-pro' ),
                    'name'              => __( 'List:', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'List ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_fluentcrm_get_lists',
                    'options_cb'        => 'automatorwp_fluentcrm_options_cb_list',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'unroll' ) {
            return sprintf( __( '%1$s %2$s from %3$s', 'automatorwp-pro' ), '{operation}', '{contact}', '{list}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        global $wpdb;

        // Shorthand
        $operation = $action_options['operation'];
        $list_id = $action_options['list'];
        $contact = $action_options['contact'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'add';
        }

        // Bail if empty list to assign
        if( empty( $list_id ) ) {
            return;
        }

        $list = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}fc_lists WHERE id = %s",
            $list_id
        ) );

        // Bail if list not exists
        if( ! $list ) {
            return;
        }

        $subscriber = FluentCrm\App\Models\Subscriber::where( 'email', $contact )->first();

        // Bail if subscriber not exists
        if( ! $subscriber ) {
            return;
        }

        switch ( $operation ) {
            case 'add':
                // Add contact to list
                $subscriber->attachLists( array( $list_id ) );
                break;
            case 'remove':
                // Remove contact from list
                $subscriber->detachLists( array( $list_id ) );
                break;
        }


    }

}

new AutomatorWP_FluentCRM_Contact_List();